<?php
/**
 * Logic for Moslate 0.3b+ (for Mambo 4.5.1)
 * @package moslate
 * @version $Id: logic.php 208 2006-09-19 22:48:33Z  $
 * @copyright (C) 2004-2006 Daniel Ecer (de.siteof.de)
 * @license GNU/GPL

 * <b>Logic</b><br/>
 * This class provides functions to mimic JSP-style tags...<br/>
 * For example to get/set properties.<br/>
 * In addition it provides EL (Expression Language) as
 * specified by the JSTL (some parts might be implemented
 * different).<br/>
 * Additional functions might be implemented directly in:
 * <br/>LogicFunctions::invoke($name, &$arguments)<br/>
 *   (this   would be necessary if you need to return a reference)<br/>
 * or you implement functions by defining a .tld and implementing the functions
 * in the defined class. (like the standard fn.tld is implemented)<br/>
 * (The XML-file is parsed and cached using the standard cache-mechanismus but
 * enabling the cache anyway.)
 *
 * (depency is used when exporting files)
 * @moslate.depency
 *   match="\..*$"
 *
 * */

global $Logic_debug, $Logic_logFile, $mosConfig_absolute_path;
$Logic_debug	= FALSE;
$Logic_logFile	= $mosConfig_absolute_path.'/.debug.log';


/**
 * This class is responsible for logic functions. Usually defined by a TLD file.
 * TODO use singelton pattern instead of static methods.
 */
class LogicFunctions {

	function getTagText(&$xmlParent, $tagName) {
		$elements	= $xmlParent->getElementsByTagName($tagName);
		if ($elements->getLength() != 1) {
			return NULL;
		}
		$xmlTag	=& $elements->item(0);
		return $xmlTag->getText();
	}

	/**
	 * This function loads a TLD file (no caching used). Used by loadTLD.
	 */
	function &doLoadTLD($tldPath) {
		global $mosConfig_absolute_path;
		if (!file_exists($tldPath)) {
			return array();
		}
		$timestamp		= filemtime($tldPath);
		// build array
		$functions	= array();
		$functions[]	= $timestamp;
		include_once( $mosConfig_absolute_path . '/includes/domit/xml_domit_lite_include.php' );
		$xmlDoc =& new DOMIT_Lite_Document();
		$xmlDoc->resolveErrors( FALSE );
		if ($xmlDoc->loadXML($tldPath, FALSE, TRUE)) {
			$elements	= $xmlDoc->getElementsByTagName("function");
			$length		= $elements->getLength();
			for ($iFunction = 0; $iFunction < $length; $iFunction++) {
				$xmlFunction		=& $elements->item($iFunction);
				$name				= trim(LogicFunctions::getTagText($xmlFunction, 'name'));
				$functionClass		= trim(LogicFunctions::getTagText($xmlFunction, 'function-class'));
				$functionSignature	= trim(LogicFunctions::getTagText($xmlFunction, 'function-signature'));
				$t					= $functionSignature;
				$resultType	= Logic::getNextToken($t, array(' '));
				$t	= trim($t);
				$functionName	= trim(Logic::getNextToken($t, array('(')));
				$t	= trim($t);
				$t				= Logic::getNextToken($t, array(')'));
				$t				= trim($t);
				$a				= explode(',', $t);
				$argc			= count($a);
				if (($argc == 1) && (trim($a[0]) == '')) {
					$argc	= 0;
				}
				$functionClass	= str_replace('.', '_', $functionClass);
				$functions[$name]	= array($functionClass, $functionName, $argc, $argc);
				//echo '[f:'.$resultType.'/'.$functionName.'/'.$argc.'/'.$t.']<br/>';
				//echo '[name:'.htmlentities($name).']<br/>';
			}
		}
		return $functions;
	}


	/**
	 * This function attempts to load a TLD file and uses a cache if available.
	 * @since version 0.5.5
	 */
	function &loadTLD($tldPath) {
		global $mosConfig_absolute_path, $mosConfig_caching, $mosConfig_cachetime;
		if (!file_exists($tldPath)) {
			return array();
		}
		// do some caching, the standard caching is not very effective for our purpose... change the settings (if this does not work it should just slow down)
		$old_mosConfig_caching		= $mosConfig_caching;
		$old_mosConfig_cachetime	= $mosConfig_cachetime;
		$mosConfig_caching			= TRUE;
		$mosConfig_cachetime		= 100 * 24 * 60 * 60;	// a life time of 100 days since we are checking the timestamp anyway
		//$cacheId		= basename($tldPath);
		$timestamp		= filemtime($tldPath);
		$cache	=& mosCache::getCache('logic_tlds');
		$mosConfig_caching			= $old_mosConfig_caching;
		$mosConfig_cachetime		= $old_mosConfig_cachetime;

		$functions	= $cache->call('LogicFunctions::doLoadTLD', $tldPath);

		if ((is_array($functions)) && (isset($functions[0])) && ($functions[0] == $timestamp)) {
			return $functions;
		}

		// we got it from the cache but the file timestamp did not match, clean and reload it...
		$cache->clean('logic_tlds');
		$functions	= $cache->call('LogicFunctions::doLoadTLD', $tldPath);
		return $functions;
	}


	function &loadTLDs($tldDirectory) {
		global $mosConfig_absolute_path;

		// process any tld's
		$tlds		= array();
		$tldFiles	= mosReadDirectory( $tldDirectory, "\.tld$" );
		sort( $tldFiles );
		foreach ($tldFiles as $tldFile) {
			$shortname	= substr($tldFile, 0, strrpos($tldFile, '.'));
			$tlds[$shortname]	=& LogicFunctions::loadTLD($tldDirectory.'/'.$tldFile);
		}
		return $tlds;
	}


	function &invoke($name, &$arguments) {
		global $mosConfig_absolute_path;
		static $tlds=0;
		$result	= NULL;
		$argc	= count($arguments);
		switch($name) {
			// your special functions
			case 'test:new':
				if ($argc == 0) {
					trigger_error('EL:too few parameters for '.$name, E_USER_WARNING);
					return $result;
				}
				if ($argc > 1) {
					trigger_error('EL:too many parameters for '.$name, E_USER_WARNING);
					return $result;
				}
				$result	=& Logic::newInstance($arguments[0]);
				break;
			default:
				// try to find in our tlds
				$i = strpos($name, ':');
				if (!is_array($tlds)) {
					$tldDirectory		= $mosConfig_absolute_path.'/components/com_moslate/includes/logic/';
					$tlds	= LogicFunctions::loadTLDs($tldDirectory);
				}
				if ((is_array($tlds)) && ($i !== FALSE) && ($i > 0) && ($i+1 < strlen($name))) {
					$namespace	= substr($name, 0, $i);
					$alias		= substr($name, $i+1);
					//echo '['.$namespace.'/'.$alias.']';
					if (isset($tlds[$namespace])) {
						if (isset($tlds[$namespace][$alias])) {
							$a	= $tlds[$namespace][$alias];
							if (count($a) == 4) {
								$functionClass	= $a[0];
								$functionName	= $a[1];
								$minArgCount	= $a[2];
								$maxArgCount	= $a[3];
								if ($argc < $minArgCount) {
									trigger_error('EL:too few parameters for '.$name.' ('.$minArgCount.'-'.$maxArgCount.' expected, got '.$argc.')', E_USER_WARNING);
									return $result;
								}
								if ($argc > $maxArgCount) {
									trigger_error('EL:too many parameters for '.$name.' ('.$minArgCount.'-'.$maxArgCount.' expected, got '.$argc.')', E_USER_WARNING);
									return $result;
								}
								if (!class_exists($functionClass)) {
									$fileName	= $mosConfig_absolute_path.'/components/com_moslate/includes/logic/'.$functionClass.'.php';
									if (file_exists($fileName)) {
										include_once($fileName);
									}
									if (!class_exists($functionClass)) {
										trigger_error('EL:class does not exist:'.$functionClass, E_USER_WARNING);
										return $result;
									}
								}
								$f	= array($functionClass, $functionName);
								if (!is_callable($f)) {
									trigger_error('EL:function does not exist:'.$functionClass.'::'.$functionName, E_USER_WARNING);
									return $result;
								}
								$result	= call_user_func_array($f, $arguments);
								return $result;
							}
						}
					}
				}
				trigger_error('EL:unknown function:'.$name, E_USER_WARNING);
		}
		return $result;
	}
}


// TODO merge request, session and cookie list

class LogicRequestList {
	function &getItem($index) {
		$result	= mosGetParam($_REQUEST, $index);
		return $result;
	}
	function &getItems() {
		$result	= $_REQUEST;
		return $result;
	}
}

class LogicSessionList {
	function &getItem($index) {
		$result	= $_SESSION[$index];
		return $result;
	}
	function setItem($index, $value) {
		$_SESSION[$index]	= $value;
	}
	function removeItem($index) {
		unset($_SESSION[$index]);
	}
	function &getItems() {
		$result	= $_SESSION;
		return $result;
	}
}

class LogicCookieList {
	function &getItem($index) {
		$result	= $_COOKIE[$index];
		return $result;
	}
	function setItem($index, $value) {
		$_COOKIE[$index]	= $value;
	}
	function removeItem($index) {
		unset($_COOKIE[$index]);
	}
	function &getItems() {
		$result	= $_COOKIE;
		return $result;
	}
}


/**
 * This class handles variable scopes. One instance of this class is able to handle multiple scopes.
 * Variables set in a nested scope will get invalid when this scope is left again.
 */
class LogicScopeContainer {

	var $scopeArray	= array();
	var $scopeByNameMap	= array();

	/**
	 * starts a new scope...
	 * @param string name the name of the new scope (optional). Named scopes cannot be removed or overwritten.
	 */
	function &beginScope($name = NULL) {
		global $Logic_debug;
//		Logic::logDebug('beginScope:'.$name.' - '.count($this->scopeArray).'');
		$scope	= array();
//		array_push($this->scopeArray, $scope);	// removed call-by-reference
		$this->scopeArray[]		=& $scope;		// fix: need to add reference
		if ($name != NULL) {
			$this->scopeByNameMap[$name]	=& $scope;
		}
		if ($Logic_debug) {
			Logic::logDebug('beginScope: after - '.count($this->scopeArray).':'.Logic::toDebugString($this->scopeArray));
		}
		return $scope;
	}

	function &endScope() {
		global $Logic_debug;
		if ($Logic_debug) {
			Logic::logDebug('endScope: before - '.count($this->scopeArray).':'.Logic::toDebugString($this->scopeArray));
		}
		end($this->scopeArray);
		$o	=& $this->scopeArray[key($this->scopeArray)];
		array_pop($this->scopeArray);	// array_pop does not return a reference
		return $o;
	}

	function &getItem($index) {
		global $Logic_debug;
		$scopeArray	=& $this->scopeArray;
		// pageScope always refers to the last scope
		if (($index == 'pageScope') && (count($scopeArray) > 0)) {
			return $scopeArray[count($scopeArray)-1];
		}
		if (isset($this->scopeByNameMap[$index])) {
//			Logic::logDebug('getItem:'.$index.':scopeByNameMap['.$index.']');
			return $this->scopeByNameMap[$index];
		}
		for ($i = count($scopeArray)-1; $i >= 0; $i--) {
			if (isset($scopeArray[$i][$index])) {
				if ($Logic_debug) {
					Logic::logDebug('getItem:'.$index.':scopeArray['.$i.']');
				}
				return $scopeArray[$i][$index];
			}
		}
		Logic::logDebug('getItem:'.$index.' not found - scopeArray='.Logic::toDebugString($scopeArray));
		// PHP BUG #33679 (PHP 4.4/5.1): Notice: Only variable references should be returned by reference
		$temp	= NULL;
		return $temp;
	}

	function setItem($index, $value, $scope = NULL) {
		if (isset($this->scopeByNameMap[$index])) {
			return FALSE;
		}
		$scopeArray	=& $this->scopeArray;
		// uncomment if we want to update variables of a parent scope
//		for ($i = count($scopeArray)-1; $i >= 0; $i--) {
//			if (isset($scopeArray[$i][$index])) {
//				if ((is_object($value)) || (is_array($value))) {
//					$scopeArray[$i][$index]	=& $value;
//				} else {
//					$scopeArray[$i][$index]	= $value;
//				}
//				return TRUE;
//			}
//		}
//		Logic::logDebug('setItem:'.$index.':count(scopeArray)='.count($scopeArray).'');
		if (count($scopeArray) > 0) {
			if ((is_object($value)) || (is_array($value))) {
				$scopeArray[count($scopeArray)-1][$index]	=& $value;
			} else {
				unset($scopeArray[count($scopeArray)-1][$index]);	// remove any previous reference
				$scopeArray[count($scopeArray)-1][$index]	= $value;
			}
		}
		return TRUE;
	}

	function removeItem($index) {
		if (isset($this->scopeByNameMap[$index])) {
			return FALSE;
		}
		// remove entries of the top scope only
		$scopeArray	=& $this->scopeArray;
		if (count($scopeArray) > 0) {
			unset($scopeArray[count($scopeArray)-1][$index]);
		}
	}

	function &getItems() {
		$scopeArray	=& $this->scopeArray;
		$a	= array();
		foreach(array_keys($this->scopeByNameMap) as $k) {
			array_push($a, $this->scopeByNameMap[$k]);
		}
		for($i = 0; $i <= count($scopeArray); $i++) {
			$scope	=& $scopeArray[$i];
			foreach(array_keys($scope) as $k) {
				if ((is_object($scope[$k])) || (is_array($scope[$k]))) {
					array_push($a, $scope[$k]);
				} else {
					$a[]	= $scope[$k];
				}
			}
		}
		return $a;
	}
}


/**
 * This class contains functions to implement variable scopes and expression language.
 * TODO use singelton instead of static method
 */
class Logic {

	function &newBean() {
		// PHP BUG #33679 (PHP 4.4/5.1): Notice: Only variable references should be returned by reference
		$temp	= array();
		return $temp;
	}

	function logDebug($message) {
		global $Logic_debug;
		if ($Logic_debug) {
			$logger		=& Logic::getLogger();
			$logger->logDebug($message);
		}
	}

	function &getLogger() {
		global $mosConfig_absolute_path, $Logic_debug, $Logic_logFile;
		if (!$Logic_debug) {
			// PHP BUG #33679 (PHP 4.4/5.1): Notice: Only variable references should be returned by reference
			$logger	= NULL;
			return $logger;
		}
		if (!isset($GLOBALS['MoslateLogic_Logger'])) {
			if (!class_exists('mosMoslateLogger')) {
				require_once($mosConfig_absolute_path.'/components/com_moslate/moslate.logger.class.php');
			}
//			$logger	=& new mosMoslateLoggerBuffer();
			$logger	=& new mosMoslateLoggerFile($Logic_logFile);
			$GLOBALS['MoslateLogic_Logger']	=& $logger;
		}
		return $GLOBALS['MoslateLogic_Logger'];
	}

	function flushLog() {
		global $mosConfig_live_site, $Logic_debug;
		if ($Logic_debug) {
			$logger		=& Logic::getLogger();
			if (isset($logger->entries)) {
				$logger2	=& new mosMoslateLoggerHtml();
				if (!DEFINED('MOSLATE_CSS_INCLUDED')) {
					// note: this may cause the admin page to not validate, which it doesn't anyway
					echo '<link href="'.$mosConfig_live_site.'/components/com_moslate/css/moslate.css" rel="stylesheet" type="text/css"/>';
				}
				$entries	= $logger->entries;
				$logger->entries	= array();
				$logger2->logEntries($entries);
			}
		}
	}

	function &getRootBean() {
		if (!isset($GLOBALS['MoslateLogic_RootBean'])) {
//			$GLOBALS['MoslateLogic_RootBean'] = Logic::newBean();	// create empty bean
			$rootBean	=& new LogicScopeContainer();
			$applicationScope	=& $rootBean->beginScope('applicationScope');	// dummy scope, we do not have any equivalent
			$sessionScope	=& $rootBean->beginScope('sessionScope');		// TODO implement
			$requestScope	=& $rootBean->beginScope('requestScope');
			$pageScope		=& $rootBean->beginScope('pageScope');
			$requestList	=& new LogicRequestList();
			$GLOBALS['MoslateLogic_RootBean'] =& $rootBean;
			Logic::set($requestScope, 'param', $requestList);
			Logic::set($requestScope, 'paramValues', $requestList);	// TODO filtered list returning strings only
			Logic::set($requestScope, 'cookie', new LogicCookieList());
//			Logic::logDebug('requestScope:'.Logic::toDebugString($requestScope));
		}
		return $GLOBALS['MoslateLogic_RootBean'];
	}

	function beginScope() {
		$rootBean	=& Logic::getRootBean();
		$rootBean->beginScope();
	}

	function endScope() {
		$rootBean	=& Logic::getRootBean();
		$rootBean->endScope();
	}

	function &getAutoReference(&$value) {
		// PHP seem not to like references to other than objects and arrays
		if ((is_object($value)) || (is_array($value))) {
			return $value;
		}
		if (is_null($value)) {
			$copy	= NULL;
		} else if (is_string($value)) {
			$copy	= ''.$value;
		} else {
			$copy	= $value;
		}
		return $copy;
	}

	function securePathComponent($s) {
		return str_replace('..', '__', strtr($s, ':/\\', "___"));
	}

	function &newInstance($className) {
		global $mosConfig_absolute_path;
		$result	= NULL;
		if (!class_exists($className)) {
			$className	= Logic::securePathComponent($className);
			$fileName	= $mosConfig_absolute_path.'/components/com_moslate/includes/logic/'.$className.'.php';
			if (file_exists($fileName)) {
				include_once($fileName);
			}
			if (!class_exists($className)) {
				trigger_error('EL:class does not exist:'.$className, E_USER_WARNING);
				return $result;
			}
		}
		$result	=& new $className();
		return $result;
	}

	function &getObjectByParentAndName(&$parent, &$nameOrRef, $isCreate = FALSE) {
		global $Logic_debug;
		if (is_null($parent)) {
			// PHP BUG #33679 (PHP 4.4/5.1): Notice: Only variable references should be returned by reference
			$temp	= NULL;
			return $temp;
		}
		if (is_array($nameOrRef)) {
			return $nameOrRef;	// it is already the bean (array)
		}
		if (is_object($nameOrRef)) {
			return $nameOrRef;	// it is already the bean (object)
		}
		$name	= ''.$nameOrRef;
		if ($name == '') {
			return $parent;
		}
		$i = strpos('.', $name);
		if ($Logic_debug) {
			Logic::logDebug('getObjectByParentAndName: parent='.Logic::toDebugString($parent).'; name='.Logic::toDebugString($name));
		}
		$nullValue	= NULL;
		if ($i === FALSE) {
			if (is_array($parent)) {
				if (!isset($parent[$name])) {
					if ($isCreate) {
						$parent[$name]	= Logic::newBean();	// create empty bean
					} else {
						// PHP BUG #33679 (PHP 4.4/5.1): Notice: Only variable references should be returned by reference
						return $nullValue;
					}
				}
				if ($Logic_debug) {
					Logic::logDebug('getObjectByParentAndName: get from array('.Logic::toDebugString($parent).', index='.Logic::toDebugString($name).')=...');
				}
				$v	=& Logic::getAutoReference($parent[$name]);
//				$v	=& $parent[$name];
				if ($Logic_debug) {
					Logic::logDebug('getObjectByParentAndName: ...='.Logic::toDebugString($v));
				}
//				if (is_string($v)) {
//					return ''.$v;	// unfortunatelly there seem to be a bug with strings
//				}
				return $v;
				//return $parent[$name];
			}
			if (is_object($parent)) {
				if ($Logic_debug) {
					Logic::logDebug('getObjectByParentAndName: object name='.Logic::toDebugString($name));
				}
				if (method_exists($parent, 'getItem')) {
					// PHP BUG #33679 (PHP 4.4/5.1): Notice: Only variable references should be returned by reference
					$temp	=& $parent->getItem($name);
					$result	=& Logic::getAutoReference($temp);
					if (($result == NULL) && ($isCreate)) {
						if (method_exists($parent, 'setItem')) {
							$parent->setItem($name, Logic::newBean());
							$result	=& $parent->getItem($name);
						}
					}
					return $result;
				}
				if (!isset($parent->$name)) {
					if ($isCreate) {
						$parent->$name	= Logic::newBean();	// create empty bean
					} else {
						return $nullValue;
					}
				}
				// PHP BUG #33558 (PHP 4.4/5.1): Notice: Only variable references should be returned by reference
//				return Logic::getAutoReference($parent->$name);
				// workaround:
				$result	=& Logic::getAutoReference($parent->$name);
				return $result;
			}
			return $nullValue;	// parent is no valid bean
		} else {
			$name1	= substr($name, 0, $i);
			$name2	= substr($name, $i+1);
			$bean	=& Logic::getObjectByParentAndName($parent, $name1, $isCreate);
			if (is_null($bean)) {
				return $nullValue;
			}
			return Logic::getObjectByParentAndName($bean, $name1, $isCreate);
		}
	}

	function &getObjectByName(&$name, $isCreate = FALSE) {
		$parent	=& Logic::getRootBean();
		// PHP BUG #33558 (PHP 4.4/5.1): Notice: Only variable references should be returned by reference
//		return Logic::getObjectByParentAndName($parent, $name, $isCreate);
		// workaround:
		$result	=& Logic::getObjectByParentAndName($parent, $name, $isCreate);
		return $result;
	}

	function &getBean($name, $isCreate = FALSE) {
		// PHP BUG #33558 (PHP 4.4/5.1): Notice: Only variable references should be returned by reference
//		return Logic::getObjectByName($name, $isCreate);
		// workaround:
		$result	=& Logic::getObjectByName($name, $isCreate);
		return $result;
	}

	function &getObject($name, $isCreate = FALSE) {
		// PHP BUG #33558 (PHP 4.4/5.1): Notice: Only variable references should be returned by reference
//		return Logic::getObjectByName($name, $isCreate);
		// workaround:
		$result	=& Logic::getObjectByName($name, $isCreate);
		return $result;
	}

	/**
	 * Sets a property. Value is passed by reference.
	 *
	 */
	 // fix 0.5.5: name by ref
	function set(&$name, $property, &$value) {
		global $Logic_debug;
		if (is_null($value)) {
			return Logic::remove($name, $property);
		}
		$v	=& Logic::getAutoReference($value);
		if ($Logic_debug) {
			Logic::logDebug('Logic.set:'.Logic::toDebugString($name).' property='.Logic::toDebugString($property).'; value='.Logic::toDebugString($value).' ('.gettype($value).')');
		}
		if ((!is_object($name)) && (!is_array($name))) {
			$name = trim($name);
		}
		if ((!isset($property)) || ($property == '')) {
			$property = 'temp';
		}
		$property = trim($property);
//		$bean	=& Logic::getBean($name, TRUE);		// get bean uses call-by-value for name
		$bean	=& Logic::getObjectByName($name, TRUE);
		if (is_null($bean)) {
			return FALSE;
		}
		if (is_array($bean)) {
			$bean[$property]	=& $v;
			return TRUE;
		}
		if (is_object($bean)) {
			if (method_exists($bean, 'setItem')) {
				$bean->setItem($property, $v);
				return TRUE;
			}
			$bean->$property	=& $v;
			return TRUE;
		}
		return FALSE;
	}

	/**
	 * Same as set, but value is passed by value.
	 */
	function setProperty($name, $property = '', $value = '') {
		return Logic::set($name, $property, $value);
	}

	// fix: name call-by-reference
	function &get(&$name, $property = '') {
		global $Logic_debug;
		if ($Logic_debug) {
			Logic::logDebug('get('.Logic::toDebugString($name).', property='.Logic::toDebugString($property).')');
		}
		$name = trim($name);
		if ((!isset($property)) || ($property == '')) {
			$property = 'temp';
		}
		$property = trim($property);
		//echo '[getProperty('.$name.','.$property.')<br/>';
		$bean	=& Logic::getBean($name, TRUE);
		if (is_null($bean)) {
			// PHP BUG #33679 (PHP 4.4/5.1): Notice: Only variable references should be returned by reference
			$result	= NULL;
			return $result;
		}
		if (is_array($bean)) {
			if (isset($bean[$property])) {
				return $bean[$property];
			}
		}
		if (is_object($bean)) {
			if (method_exists($bean, 'getItem')) {
				// PHP BUG #33558 (PHP 4.4/5.1): Notice: Only variable references should be returned by reference
				$result	=& $bean->getItem($property);
				return $result;
			}
			if (isset($bean->$property)) {
				return $bean->$property;
			}
		}
		// PHP BUG #33679 (PHP 4.4/5.1): Notice: Only variable references should be returned by reference
		$result	= NULL;
		return $result;
	}

	function getProperty($name, $property = '') {
		return Logic::get($name, $property);
	}

	function addProperty($name, $value = '') {
		$name = trim($name);
		//echo '[addProperty('.$name.','.$value.')<br/>';
		$bean	=& Logic::getBean($name, TRUE);
		if (is_null($bean)) {
			return FALSE;
		}
		if (is_array($bean)) {
			$bean[]	= $value;
			return TRUE;
		}
		if (is_object($bean)) {
			if (method_exists($bean, 'addItem')) {
				return $bean->addItem($value);
			}
			return FALSE;
		}
		return FALSE;
	}

	function remove($name, $property) {
		return Logic::removeProperty($name, $property);
	}

	function removeProperty($name, $property = '') {
		$name = trim($name);
		if ((!isset($property)) || ($property == '')) {
			$property = 'temp';
		}
//		echo '[removeProperty('.$name.','.$property.')<br/>';
		$bean	=& Logic::getBean($name, FALSE);
		if (is_null($bean)) {
			return FALSE;
		}
		if (is_array($bean)) {
			unset($bean[$property]);
			return TRUE;
		}
		if (is_object($bean)) {
			if (method_exists($bean, 'removeItem')) {
				return $bean->removeItem($property);
			}
			return TRUE;
		}
		return FALSE;
	}

	function removePropertyByValue($name, $value = '') {
		$name = trim($name);
		//echo '[removePropertyByValue('.$name.','.$value.')<br/>';
		$bean	=& Logic::getBean($name, TRUE);
		if (is_null($bean)) {
			return FALSE;
		}
		if (is_array($bean)) {
			$keys = array_keys($bean, $value);
			foreach($keys as $key) {
				unset($bean[$key]);
			}
			return TRUE;
		}
		if (is_object($bean)) {
			if (method_exists($bean, 'removeItemByValue')) {
				return $bean->removeItem($value);
			}
			return TRUE;
		}
		return FALSE;
	}

	function removeProperties($name) {
		$name = trim($name);
		//echo '[removeProperties('.$name.')<br/>';
		$bean	=& Logic::getBean($name, TRUE);
		if (is_null($bean)) {
			return FALSE;
		}
		if (is_array($bean)) {
			array_splice($bean, 0);	// clears the array
			return TRUE;
		}
		if (is_object($bean)) {
			if (method_exists($bean, 'clear')) {
				return $bean->clear();
			}
			return FALSE;
		}
		return FALSE;
	}


	/**
	 * Evaluates the given operator with both operands as a parameter.
	 */
	function &applyOperator($operator, &$operand1, &$operand2) {
		$result	= 0;
		switch($operator) {
			case '+':
				$result	= $operand1 + $operand2;
				break;
			case '-':
				$result	= $operand1 - $operand2;
				break;
			case '*':
				$result	= $operand1 * $operand2;
				break;
			case 'div':
			case '/':
				$result	= $operand1 / $operand2;
				break;
			case 'mod':
			case '%':
				$result	= $operand1 % $operand2;
				break;
			case 'eq':
			case '==':
				$result	= $operand1 == $operand2;
				break;
			case 'ne':
			case '!=':
				$result	= $operand1 != $operand2;
				break;
			case 'le':
			case '<=':
				$result	= $operand1 <= $operand2;
				break;
			case 'ge':
			case '>=':
				$result	= $operand1 >= $operand2;
				break;
			case 'lt':
			case '<':
				$result	= $operand1 < $operand2;
				break;
			case 'gt':
			case '>':
				$result	= $operand1 > $operand2;
				break;
			case 'and':
			case '&&':
				$result	= $operand1 && $operand2;
				break;
			case 'or':
			case '||':
				$result	= $operand1 || $operand2;
				break;
			default:
				trigger_error('EL:unknown operator:'.$operator, E_USER_WARNING);
		}
//		echo 'applyOperator('.$operator.','.Logic::toDebugString($operand1).' ('.gettype($operand1).')'.'['.ord($operand1).']'.','.Logic::toDebugString($operand2).' ('.gettype($operand2).')'.')='.Logic::toDebugString($result).'<br/>';
		echo Logic::logDebug('applyOperator('.$operator.','.Logic::toDebugString($operand1).' ('.gettype($operand1).')'.'['.ord($operand1).']'.','.Logic::toDebugString($operand2).' ('.gettype($operand2).')'.')='.Logic::toDebugString($result).'');
		return $result;
	}


	/**
	 * Evaluates the given modified using the operand.
	 */
	function &applyModifiers(&$modifiers, $operand) {
		global $Logic_debug;
		$count	= count($modifiers);
		if ($count == 0) {
			return $operand;
		}
		if (is_null($operand)) {
			$result	= NULL;
		} else {
			$result	= $operand;
		}
		for ($i = $count-1; $i >= 0; $i--) {
			$modifier	= $modifiers[$i];
			switch($modifier) {
				case '+':
					break;
				case '-':
					$result	= (-$result);
					break;
				case 'not':
				case '!':
					$result	= (!$result);
					break;
				case 'empty':
					if (is_null($result)) {
						$result	= TRUE;
					} else  if (is_array($result)) {
						$result	= (count($result) == 0);
					} else  if (is_object($result)) {
						if (method_exists($result, 'isEmpty')) {
							$result	= $result->isEmpty();
						} else {
							$result	= FALSE;
						}
					} else {
						$result	= ($result == '');
					}
					break;
				default:
					trigger_error('EL:unknown modifier:'.$modifier, E_USER_WARNING);
			}
		}
		if ($Logic_debug) {
			echo Logic::logDebug('applyModifiers('.Logic::toDebugString($modifiers).', operand='.Logic::toDebugString($operand).' ('.gettype($operand).')'.')='.Logic::toDebugString($result));
		}
		array_splice($modifiers, 0);
		// PHP BUG #33558 (PHP 4.4/5.1): Notice: Only variable references should be returned by reference
//		return Logic::getAutoReference($result);
		// workaround:
		$result2	= Logic::getAutoReference($result);
		return $result2;
	}


	/**
	 * Appends the array values from a to result. If a value its values will get added instead (recursively)
	 */
	function flattenArray(&$result, $a) {
		$c		= count($a);
		for($i=0; $i < $c; $i++) {
			$v	=& $a[$i];
			if (is_array($v)) {
				Logic::flattenArray($result, $v);
			} else {
				$result[]	= $v;
			}
		}
		return $result;
	}


	/**
	 * Evaluates the inner part of a expression (that is without the surrounding "${...}")
	 * This is the core and most complex part of the EL implementation.
	 */
	function &evalInnerEL($text) {
		global $Logic_debug;
		static $constants, $operatorKeywordsByPrecedence, $operatorTokensByPrecedence,
			$modifierKeywords, $modifierTokens, $quoteTokens, $otherTokens,
			$operatorTokens, $operatorTokens2, $separatingTokens,
			$keywordTokens, $operatorTokensByPrecedence2,
			$unaryOperators;
		if (!is_array($constants)) {
			$constants						= array('null' => NULL, 'false' => FALSE, 'true' => TRUE);
//				'param' => new LogicRequestList(), 'requestScope' => new LogicRequestList(),
//				'sessionScope' => new LogicRequestList(), 'cookie' => new LogicCookieList());
			$operatorKeywordsByPrecedence	= array(array('or'), array('and'), array('eq', 'ne', 'lt', 'gt', 'le', 'ge'), array(), array('div', 'mod'));
			$operatorTokensByPrecedence		= array(array('||'), array('&&'), array('==', '!=', '<=', '>=', '<', '>'), array('+', '-'), array('*', '/', '%'));
			$unaryOperators					= array('+', '-');	// will be used as modificators
			$modifierKeywords				= array('not', 'empty');
			$modifierTokens					= array('!');
			$quoteTokens					= array("\"", '\'');
			$otherTokens					= array('(', '[', ']', '.');
			// convert for better access
			$operatorTokens					= array();
			Logic::flattenArray($operatorTokens, $operatorTokensByPrecedence);
			$operatorTokens2				= $operatorTokens;
			Logic::flattenArray($operatorTokens2, $operatorKeywordsByPrecedence);
			$separatingTokens				= $otherTokens;
			$separatingTokens				= array_merge($separatingTokens, $quoteTokens);
			$separatingTokens				= array_merge($separatingTokens, $operatorTokens);
			$separatingTokens				= array_merge($separatingTokens, $modifierTokens);
			$keywordTokens					= $modifierKeywords;
			Logic::flattenArray($keywordTokens, $operatorKeywordsByPrecedence);
			$operatorTokensByPrecedence2	= $operatorKeywordsByPrecedence;
			for($i = 0; $i < count($operatorTokensByPrecedence2); $i++) {
				$operatorTokensByPrecedence2[$i]	=
					array_merge($operatorTokensByPrecedence2[$i], $operatorTokensByPrecedence[$i]);
			}
			//print_r($separatingTokens);
			//print_r($keywordTokens);
		}

		$debug	= $Logic_debug;

		if ($debug) {
			Logic::logDebug('evalInnerEL('.Logic::toDebugString($text).')');
		}

		$srcText			= $text;
		$operands			= array();
		$operators			= array();
		$isExpectOperator	= FALSE;
		$modifiers			= array();
		$isTempOperand		= FALSE;
		$tempOperand		= NULL;
		$isAwaitProperty	= FALSE;
		$nullValue			= NULL;	// this value is used to be able to return a NULL value as a reference
		//echo '[evalInnerEL:'.$text.']';
		$whitespaces	= " \t\r\n";

		// handle: <condition> ? <then> : <else>
		$i	= Logic::findTermEndSkipQuotes($text, '', '?', '');
		if ($i !== FALSE) {
			$j	= Logic::findTermEndSkipQuotes($text, '', '(', '');
			if (($j === FALSE) || ($j > $i)) {
				$iDepth		= 1;
				$tempPos	= $i+1;
				while($iDepth > 0) {
					$j	= Logic::findTermEndSkipQuotes($text, '', ':', '', $tempPos);
					if ($j === FALSE) {
						trigger_error('EL: \':\' expected', E_USER_WARNING);
						return NULL;
					}
					$iDepth--;
					$k	= Logic::findTermEndSkipQuotes($text, '', '?', '', $tempPos);
					if (($k !== FALSE) && ($k < $j)) {
						$iDepth++;
					}
					$tempPos	= $j+1;
				}
				$conditionStr	= substr($text, 0, $i);
				$ifStr			= substr($text, $i+1, $j-$i-1);
				$thenStr		= substr($text, $j+1);
				// PHP BUG #33679 (PHP 4.4/5.1): Notice: Only variable references should be returned by reference
				// workaround: use variable
				if (Logic::toBool(Logic::evalInnerEL($conditionStr))) {
					$result	=& Logic::evalInnerEL($ifStr);
				} else {
					$result	=& Logic::evalInnerEL($thenStr);
				}
				return $result;
			}
			// it will be handled when the breaket will be resolved
		}

		while(TRUE) {
			$text	= trim($text);
			if ($text == '') {
				break;
			}
			//echo "[text:".$text."/".count($operands)."/".count($operators)."]<br/>";
			$token	= '';
			$preToken	= '';
			$i		= FALSE;
			foreach($separatingTokens as $separatingToken) {
				$j	= strpos($text, $separatingToken);
				if (($j !== FALSE) && (($i === FALSE) || ($j < $i))) {
					$i		= $j;
					$token	= $separatingToken;
				}
			}
			$i2		= $i;
			for ($index = 0; $index < strlen($whitespaces); $index++) {
				$j	= strpos($text, $whitespaces{$index});
				if (($j !== FALSE) && (($i2 === FALSE) || ($j < $i2))) {
					$i2		= $j;
				}
			}
			$i3		= $i2;
			while(($i3 < $i) && (strpos($whitespaces, $text{$i3}) !== FALSE)) {
				$i3++;
			}
			if ($i3 != $i) {
				$token	= ' ';	// the space(s) where not unnecessary
				$i		= $i2;
			}

			if ($i !== FALSE) {
				$preToken	= trim(substr($text, 0, $i2));
				if ($preToken != '') {
					foreach($keywordTokens as $keyword) {
						if ($preToken == $keyword) {
							$i		= 0;
							$token	= $preToken;
							$preToken	= '';
							break;
						}
					}
				}
				//echo "[token:".$i."/".$token."/".$isExpectOperator."]";
			} else {
				$preToken	= $text;
			}

			if ($debug) {
				Logic::logDebug('evalInnerEL: token='.Logic::toDebugString($token).'; preToken='.
						Logic::toDebugString($preToken).'; isAwaitProperty='.Logic::toDebugString($isAwaitProperty));
			}

			if ($isAwaitProperty) {
				if (!$isTempOperand) {
					trigger_error('EL:internal parsing error (when found:'.$preToken.'/'.$token.')', E_USER_WARNING);
					return $nullValue;
				}
				if ($preToken == '') {
					trigger_error('EL:property name expected (found:'.$preToken.'/'.$token.'/'.$text.')', E_USER_WARNING);
					return $nullValue;
				}
				if ($token == '(') {
					$isAwaitProperty	= FALSE;	// we will try to call a function instead
				} else {
					// PHP BUG: the following line would cause PHP to crash or trying to allocate immense amount of memory
//					$tempOperand	=& Logic::getObjectByParentAndName($tempOperand, $preToken);
					$newTempOperand	=& Logic::getObjectByParentAndName($tempOperand, $preToken);
					unset($tempOperand);
					$tempOperand	=& $newTempOperand;
					unset($newTempOperand);
					$isAwaitProperty	= FALSE;
					if ($i !== FALSE) {
						$text	= substr($text, $i);
						continue;
					} else {
						break;
					}
				}
			}

			if ($isTempOperand) {
				if (count($operands) != count($operators)) {
					trigger_error('EL:internal parsing error (when found:'.$preToken.'/'.$token.')', E_USER_WARNING);
					return $nullValue;
				}
				if (in_array($token, $operatorTokens2)) {
					if ($debug) {
						Logic::logDebug('applyModifiers: '.Logic::toDebugString($modifiers).' operand='.Logic::toDebugString($tempOperand));
					}
					$operands[]	=& Logic::applyModifiers($modifiers, $tempOperand);
					$tempOperand	= NULL;
					$isTempOperand	= FALSE;
					$isExpectOperator	= TRUE;
				} else if (($token == '[') || ($token == '.') || ($token == '(')) {
					// we will process this somewhere below
					$isExpectOperator	= FALSE;
				} else {
					// this will trigger operator expected
					$isExpectOperator	= TRUE;
				}
			} else {
				$isExpectOperator	= (count($operands) > count($operators));
			}

			if ($isExpectOperator) {
				if ($i === FALSE) {
					trigger_error('EL:operator expected (found:'.$preToken.')', E_USER_WARNING);
					return $nullValue;
				} else {
					if (!in_array($token, $operatorTokens2)) {
						trigger_error('EL:operator expected (found:'.$preToken.'/'.$token.')', E_USER_WARNING);
						return $nullValue;
					} else {
						if ($i != 0) {
							trigger_error('EL:operator expected (found:'.$preToken.')', E_USER_WARNING);
							return $nullValue;
						}
						$operators[]		= $token;
						$i	+= strlen($token);
					}
				}
			} else {
				if ($i === FALSE) {
					//echo '[add operand:'.$text.']';
					$lowerText	= strtolower(trim($text));
					if (isset($constants[$lowerText])) {
//						Logic::logDebug('applyModifiers: '.$modifiers.' constants['.$lowerText.']');
						$operands[]	= Logic::applyModifiers($modifiers, $constants[$lowerText]);
					} else if ($text == 'null') {	// isset returns FALSE
//						Logic::logDebug('applyModifiers: '.$modifiers.' NULL');
						$operands[]	= Logic::applyModifiers($modifiers, NULL);
					} else if (strpos($text, ' ') !== FALSE) {
						trigger_error('EL:operator missing', E_USER_WARNING);
						return $nullValue;
					} else if (is_numeric($text)) {
						if ($debug) {
							Logic::logDebug('applyModifiers: '.Logic::toDebugString($modifiers).' numeric='.Logic::toDebugString($text).'');
						}
						$operands[]	= Logic::applyModifiers($modifiers, floatval($text));
						if ($debug) {
							Logic::logDebug('operands: '.Logic::toDebugString($operands));
						}
					} else {
						//echo 'eval!:'.$lowerText.']';
						if ($debug) {
							Logic::logDebug('applyModifiers: '.Logic::toDebugString($modifiers).' object('.Logic::toDebugString($text).')');
						}
						$operands[]	=& Logic::applyModifiers($modifiers, Logic::getObject($text, FALSE));
					}
				} else {
					if ($token == '(') {
						$j	= Logic::findTermEndSkipQuotes($text, '(', ')', '', $i+1);
						if ($debug) {
							Logic::logDebug('findTermEndSkipQuotes: text='.Logic::toDebugString($text).', index='.$i.', result='.$j);
						}
						if ($j === FALSE) {
							trigger_error('EL:\')\' expected, text='.$text, E_USER_WARNING);
							return $nullValue;
						}
						$funcName	= $preToken;
						$argStr		= substr($text, $i+1, $j-$i-1);
						$args		= array();
						while(TRUE) {
							$k	= Logic::findTermEndSkipQuotesAndBreakets($argStr, '', ',', '');
							if ($k === FALSE) {
								$args[]	=& Logic::evalInnerEL($argStr);
								break;
							}
							$args[]	=& Logic::evalInnerEL(substr($argStr, 0, $k));
							$argStr	= substr($argStr, $k+1);
						}
						//echo '[add function operand...]';
						if ($funcName == '') {
							$c = count($args);
							if ($c == 0) {
								trigger_error('EL:empty bracket', E_USER_WARNING);
								return $nullValue;
							} else if ($c != 1) {
								trigger_error('EL:comma not expected', E_USER_WARNING);
								return $nullValue;
							} else {
								Logic::logDebug('applyModifiers: '.Logic::toDebugString($modifiers).' arg[0]='.Logic::toDebugString($args[0]).']');
								$operands[]	=& Logic::applyModifiers($modifiers, $args[0]);
							}
						} else {
							if ($isTempOperand) {
								if (!is_object($tempOperand)) {
									trigger_error('EL:function call on a non-object ('.Logic::toString($tempOperand).')', E_USER_WARNING);
									return $nullValue;
								}
								$f	= array($tempOperand, $funcName);
								if (!is_callable($f)) {
									trigger_error('EL:function does not exist:'.$funcName, E_USER_WARNING);
									return $nullValue;
								}
								$result	= call_user_func_array($f, $args);
								if ($debug) {
									Logic::logDebug('function result: '.Logic::toDebugString($funcName));
								}
								$operands[]	= $result;
								$isTempOperand	= FALSE;
							} else {
								if ($debug) {
									Logic::logDebug('applyModifiers: '.Logic::toDebugString($modifiers).' funcName='.Logic::toDebugString($funcName).']');
								}
								$operands[]	=& Logic::applyModifiers($modifiers, LogicFunctions::invoke($funcName, $args));
							}
						}
						$text	= substr($text, $j+1);
						continue;
					} else if ($token == '[') {
						$j	= Logic::findTermEndSkipQuotes($text, '[', ']', '', $i+1);
						if ($j === FALSE) {
							trigger_error("EL:']' expected", E_USER_WARNING);
							return $nullValue;
						}
						if ($isTempOperand) {
							if ($preToken != '') {	// TODO was $preItem?
								trigger_error("EL:'.' expected", E_USER_WARNING);
								return $nullValue;
							}
						} else {
							$tempOperand	=& Logic::evalInnerEL($preToken);
						}
						$s		= substr($text, $i+1, $j-$i-1);
						$index	= Logic::toString(Logic::evalInnerEL($s));
						if ($debug) {
							Logic::logDebug('evalInnerEL: getObjectByParentAndName('.Logic::toDebugString($tempOperand).', index='.Logic::toDebugString($index).')=...');
						}
						// PHP BUG: the following line would cause PHP to crash or trying to allocate immense amount of memory
//						$tempOperand	=& Logic::getObjectByParentAndName($tempOperand, $index);
//						$tempOperand	=& Logic::getAutoReference(Logic::getObjectByParentAndName($tempOperand, $index));
						$newTempOperand	=& Logic::getObjectByParentAndName($tempOperand, $index);
						unset($tempOperand);
						$tempOperand	=& $newTempOperand;
						unset($newTempOperand);
						if ($debug) {
							Logic::logDebug('evalInnerEL: ... ='.Logic::toDebugString($tempOperand));
						}
						$isTempOperand	= TRUE;
						$i	= $j + 1;
					} else if ($token == '.') {
						if ($debug) {
							Logic::logDebug('token \'.\': isTempOperand='.$isTempOperand.'; preItem='.(isset($preItem) ? $preItem : 'n/a').'; preToken='.$preToken);
						}
						if ($isTempOperand) {
							if ($preToken != '') {	// TODO was $preItem?
								trigger_error("EL:'.' expected", E_USER_WARNING);
								return $nullValue;
							}
						} else {
							if ($debug) {
								Logic::logDebug('tempOperand=evalInnerEL('.Logic::toDebugString($preToken).')');
							}
							$tempOperand	=& Logic::evalInnerEL($preToken);
						}
						$isTempOperand		= TRUE;
						$isAwaitProperty	= TRUE;
						$i	+= strlen($token);
					} else if (in_array($token, $quoteTokens)) {
						if ($i != 0) {
							trigger_error('EL:operator missing', E_USER_WARNING);
							return $nullValue;
						} else {
							$j	= Logic::findTermEnd($text, $token, $token, '\''.$token, $i+strlen($token));
							if ($j === FALSE) {
								trigger_error("EL:expected '".$token."'", E_USER_WARNING);
								return $nullValue;
							}
							//echo '[add string operand...]';
							if ($debug) {
								Logic::logDebug('applyModifiers: '.Logic::toDebugString($modifiers).' text=...');
							}
							$operands[]	= Logic::applyModifiers($modifiers, stripcslashes(substr($text, $i+strlen($token), $j-$i-2*strlen($token)+1)));
							$i	= $j + strlen($token);
						}
					} else if (in_array($token, $operatorTokens2)) {
						if ($i == 0) {
							if (in_array($token, $unaryOperators)) {
								$modifiers[]	= $token;
								$i	+= strlen($token);
							} else {
								trigger_error('EL:operand expected (found:'.$token.')', E_USER_WARNING);
								return $nullValue;
							}
						} else {
							//echo '[add operand...]';
							//$operands[]	= Logic::getObject(substr($text, 0, $i), FALSE);
							if ($debug) {
								Logic::logDebug('applyModifiers: '.Logic::toDebugString($modifiers).' eval inner...');
							}
							$operands[]	=& Logic::applyModifiers($modifiers, Logic::evalInnerEL(substr($text, 0, $i), FALSE));
						}
					} else if ((in_array($token, $modifierTokens)) ||
						(in_array($token, $modifierKeywords))) {
						if ($i == 0) {
							$modifiers[]	= $token;
							$i	+= strlen($token);
						} else {
							if ($debug) {
								Logic::logDebug('applyModifiers: '.Logic::toDebugString($modifiers).' eval inner2');
							}
							$operands[]	=& Logic::applyModifiers($modifiers, Logic::evalInnerEL(substr($text, 0, $i), FALSE));
						}
					} else if ($token == ' ') {
						$operands[]	=& Logic::applyModifiers($modifiers, Logic::evalInnerEL(substr($text, 0, $i), FALSE));
					} else {
						trigger_error('EL:unknown token:'.$token, E_USER_WARNING);
						return $nullValue;
					}
				}
			}
			if ($i === FALSE) {
				break;
			} else {
				$text	= substr($text, $i);
			}
		}
		if ($isAwaitProperty) {
			trigger_error('EL:property name expected', E_USER_WARNING);
			return $nullValue;
		}
		if ($isTempOperand) {
			if ($debug) {
				Logic::logDebug('applyModifiers: '.Logic::toDebugString($modifiers).' tempOperand='.Logic::toDebugString($tempOperand).']');
			}
			$operands[]	=& Logic::applyModifiers($modifiers, $tempOperand);
		}
		if (count($operands) == 0) {
			return '';
		}
		//echo '[result of:'.$srcText.']';
		//print_r($operands);
		//print_r($operators);
		//echo "<br/>";
		if (count($operands) != count($operators)+1) {
			trigger_error('EL:operand expected', E_USER_WARNING);
			return $nullValue;
		}
		$c	= count($operators);
		for ($i = count($operatorTokensByPrecedence2)-1; $i >= 0; $i--) {
			$j	= 0;
			while ($j < $c) {
				if (in_array($operators[$j], $operatorTokensByPrecedence2[$i])) {
//					$operands[$j]	=& Logic::applyOperator($operators[$j], $operands[$j], $operands[$j+1]);
					if ($debug) {
						Logic::logDebug('operand['.$j.']='.$operands[$j]);
					}
					$operatorResult	=& Logic::applyOperator($operators[$j], $operands[$j], $operands[$j+1]);
					if ($debug) {
						Logic::logDebug('operatorResult='.$operatorResult);
					}
//					unset($operands[$j]);	// this does not work
					if ((is_object($operatorResult)) || (is_array($operatorResult))) {
						$operands[$j]	=& $operatorResult;
					} else {
						$operands[$j]	=& $operatorResult;	// if we do not use reference here, we might overwrite the value to the previous reference
					}
//					unset($operatorResult);
					array_splice($operands, $j+1, 1);
					array_splice($operators, $j, 1);
					$c--;
				} else {
					$j++;
				}
			}
		}
		if ($debug) {
			Logic::logDebug('evalInnerEL: result='.$operands[0]);
		}
		return $operands[0];
	}


	/**
	 * Returns a string representation of the object that gives a bit more detailed information.
	 */
	function toDebugString($obj) {
//		Logic::logDebug('toDebugString: obj='.$obj);
		if (is_bool($obj)) {
			return ($obj ? 'true' : 'false');
		}
		if (is_object($obj)) {
			if (method_exists($obj, 'toString')) {
				return $obj->toString();
			}
			return 'object '.$obj.' (class '.gettype($obj).')';
		}
		if (is_array($obj)) {
			$s		= 'array(';
			$first	= TRUE;
			foreach(array_keys($obj) as $k) {
				if (!$first) {
					$s		.= ', ';
				} else {
					$first	= FALSE;
				}
				$s		.= Logic::toDebugString($k).' => '.Logic::toDebugString($obj[$k]);
			}
			$s	.= ')';
			return $s;
		}
		if (is_string($obj)) {
			return '\''.addslashes($obj).'\'';
		}
		return ''.$obj;
	}


	/**
	 * Returns a string representation of the object that should represent the actual value.
	 */
	function toString($obj) {
		if (is_bool($obj)) {
			return ($obj ? 'true' : 'false');
		}
		if (is_object($obj)) {
			if (method_exists($obj, 'toString')) {
				return $obj->toString();
			}
		}
		return ''.$obj;
	}


	/**
	 * Returns a boolean value for the given value.
	 */
	function toBool($obj) {
		if (is_bool($obj)) {
			return ($obj ? TRUE : FALSE);
		}
		if (is_string($obj)) {
			$s	= strtolower($obj);
		} else {
			$s	= strtolower(Logic::toString($obj));
		}
		if ($s == 'true') {
			return TRUE;
		}
		if ($s == 'false') {
			return FALSE;
		}
		return $obj != FALSE;	// no strict compare here
	}


	/**
	 * Returns an array (reference) for the given object.
	 * If the parameter is an array itself the parameter will be returned.
	 * If the parameter is a list object it will return it's values.
	 * If the parameter is null, an empty array will be returned.
	 * Otherwise an array with one element, the parameter, will be returned.
	 */
	function &toArray(&$obj) {
		global $Logic_debug;
		if (is_array($obj)) {
			return $obj;
		}
		if (is_object($obj)) {
			if (method_exists($obj, 'getItems')) {
				$result	= $obj->getItems();
				return $result;	// returning directly might cause a crash
			}
		}
		if (is_null($obj)) {
			return array();
		}
		return array($obj);
	}

	function getNextToken(&$text, $separatingTokens) {
		$i		= FALSE;
		foreach($separatingTokens as $separatingToken) {
			$j	= strpos($text, $separatingToken);
			if (($j !== FALSE) && (($i === FALSE) || ($j < $i))) {
				$i		= $j;
			}
		}
		if ($i !== FALSE) {
			$result	= substr($text, 0, $i);
			$text	= substr($text, $i);
		} else {
			$result	= $text;
			$text	= '';
		}
		return $result;
	}

	function findTermEnd($text, $startStr, $endStr, $escapeStr = '', $startPos = 0) {
		$i	= FALSE;
		$k	= FALSE;
		if ($endStr == '') {
			trigger_error('findTermEnd: endStr required (text='.$text.')', E_USER_WARNING);
			return FALSE;
		}
		while(TRUE) {
			if (($startStr != '') && ($startStr != $endStr)) {
				$i = strpos($text, $startStr, $startPos);
			}
			if ($startPos > strlen($text)) {
				trigger_error('findTermEnd: startPos='.$startPos.' length='.strlen($text).' endStr='.$endStr.' text='.$text, E_USER_WARNING);
				return FALSE;
			}
			$j = strpos($text, $endStr, $startPos);
			if ($escapeStr != '') {
				$k = strpos($text, $escapeStr, $startPos);
			}
			if ($j === FALSE) {
				return FALSE;
			}
			if (($k !== FALSE) && (($k <= $j) || (($i !== FALSE) && ($k < $i)))) {
				$startPos	= $k + strlen($escapeStr);
				continue;
			}
			if (($i !== FALSE) && ($i <= $j)) {
				// TODO do not use the mosMoslateParser class here
				$startPos	= mosMoslateParser::findTermEnd($text, $startStr, $endStr, $escapeStr, $i + strlen($startStr));
				if ($startPos === FALSE) {
					return FALSE;
				}
				$startPos	+= strlen($endStr);
				continue;
			}
			return $j;
		}
	}

	function findTermEndSkipQuotes($text, $startStr, $endStr, $escapeStr = '', $startPos = 0) {
		while(TRUE) {
			$i = Logic::findTermEnd($text, $startStr, $endStr, $escapeStr, $startPos);
			if ($i !== FALSE) {
				$j	= strpos($text, '\'', $startPos);
				$k	= strpos($text, "\"", $startPos);
				if (($k !== FALSE) && (($j === FALSE) || ($k < $j))) {
					$j	= $k;
				}
				if (($j === FALSE) || ($j > $i)) {
					return $i;
				}
				if ($startStr != '') {
					$firstStartStrIndex	= strpos($text, $startStr, $startPos);
					if (($firstStartStrIndex !== FALSE) && ($firstStartStrIndex < $j)) {
						$startPos	= Logic::findTermEndSkipQuotes($text, $startStr, $endStr, $escapeStr, $firstStartStrIndex+1);
						if ($startPos === FALSE) {
							return FALSE;
						}
						$startPos++;
						continue;
					}
				}
				$quoteChar	= $text{$j};
				$startPos	= Logic::findTermEnd($text, $quoteChar, $quoteChar, '\\'.$quoteChar, $j+1);
				if ($startPos === FALSE) {
					return FALSE;
				}
				$startPos++;
				continue;
			}
			return $i;
		}
	}

	function findTermEndSkipQuotesAndBreakets($text, $startStr, $endStr, $escapeStr = '', $startPos = 0) {
		while(TRUE) {
			$i = Logic::findTermEndSkipQuotes($text, $startStr, $endStr, $escapeStr, $startPos);
			if ($i !== FALSE) {
				$j	= strpos($text, '(', $startPos);
				$k	= strpos($text, '[', $startPos);
				if (($k !== FALSE) && (($j === FALSE) || ($k < $j))) {
					$j	= $k;
				}
				if (($j === FALSE) || ($j > $i)) {
					return $i;
				}
				$breaketBegin	= $text{$j};
				if ($breaketBegin == '(') {
					$breaketEnd	= ')';
				} else {
					$breaketEnd	= ']';
				}
				$startPos	= Logic::findTermEndSkipQuotes($text, $breaketBegin, $breaketEnd, '', $j+1);
				if ($startPos === FALSE) {
					return FALSE;
				}
				$startPos++;
				continue;
			}
			return $i;
		}
	}


	/**
	 * Evaluates the given text according to the EL specification.
	 * (expressions are defined using "${...}")
	 */
	function &evalEL($text) {
		global $Logic_debug;
		//echo 'evalEL('.$text.')';
		$i	= strpos($text, '${');
		if ($i === FALSE) {
			return $text;
		}
		$j	= Logic::findTermEndSkipQuotes($text, '{', '}', '', $i+2);
		if ($j === FALSE) {
			trigger_error('EL:\'}\' expected', E_USER_WARNING);
			return $text;
		}
		$result	=& Logic::evalInnerEL(substr($text, $i+2, $j-$i-2));
		if (($i > 0) || ($j+1 != strlen($text))) {
			if ($Logic_debug) {
				Logic::logDebug('evalEL: result1='.$result);
			}
			// PHP BUG #33679 (PHP 4.4/5.1): Notice: Only variable references should be returned by reference
			$temp	= substr($text, 0, $i).Logic::toString($result).
				Logic::toString(Logic::evalEL(''.substr($text, $j+1)));
			if ($Logic_debug) {
				Logic::logDebug('evalEL: result2='.$temp);
			}
			return $temp;
		} else {
			if ($Logic_debug) {
				Logic::logDebug('evalEL: result='.$result);
			}
			return $result;
		}
	}


	/**
	 * Same as evalEL but converts the result to a string.
	 */
	function evalELAsString($text) {
		// PHP5.1 bug, does not allow to pass reference directly
//		return Logic::toString(Logic::evalEL($text));
		$result	=& Logic::evalEL($text);
		return Logic::toString($result);
	}


	/**
	 * Same as evalEL but converts the result to a boolean.
	 */
	function evalELAsBool($text) {
		// PHP5.1 bug, does not allow to pass reference directly
//		return Logic::toBool(Logic::evalEL($text));
		Logic::logDebug('evalELAsBool');
		$result	=& Logic::evalEL($text);
		$temp	= Logic::toBool($result);
		Logic::logDebug('evalELAsBool: temp='.$temp);
		return $temp;
//		return Logic::toBool($result);
	}


	/**
	 * Same as evalEL but converts the result to an array.
	 */
	function &evalELAsArray($text) {
		// PHP5.1 bug, does not allow to pass reference directly
//		return Logic::toArray(Logic::evalEL($text));
		$result	=& Logic::evalEL($text);
		return Logic::toArray($result);
	}


	// helper functions until i have a better solution (like contexts)
	// actually now contexts are supported by using Logic::beginContext() and Logic::endContext()
	function &getStackedVarArray($id) {
		$key	= 'MoslateLogic_StackedVar_'.$id;
		if (!isset($GLOBALS[$key])) {
			$GLOBALS[$key] = array();
		}
		return $GLOBALS[$key];
	}

	/**
	 * @since version 0.5.2
	 */
	function pushStackedVarRef($id, &$value) {
		$a	=& Logic::getStackedVarArray($id);
		array_push($a, $value);
	}

	function pushStackedVar($id, $value = 0) {
		$a	=& Logic::getStackedVarArray($id);
		array_push($a, $value);
	}

	/**
	 * @since version 0.5.2
	 */
	function &popStackedVarRef($id) {
		$a	=& Logic::getStackedVarArray($id);
		$c	= count($a);
		if ($c < 1) {
			trigger_error('StackedVar: no var to pop', E_USER_WARNING);
			return 0;
		}
		end($a);
		$result	=& $a[key($a)];
		array_pop($a);	// array_pop does not return a reference
		return $result;
	}

	function popStackedVar($id) {
		return Logic::popStackedVarRef($id);
	}

	/**
	 * @since version 0.5.2
	 */
	function &peekStackedVarRef($id) {
		$a	=& Logic::getStackedVarArray($id);
		$c	= count($a);
		if ($c < 1) {
			trigger_error('StackedVar: no var to peek', E_USER_WARNING);
			return 0;
		}
		return $a[$c-1];
	}

	function peekStackedVar($id) {
		return Logic::peekStackedVarRef($id);
	}

	/**
	 * @since version 0.5.2
	 */
	function setStackedVarRef($id, &$value) {
		$a	=& Logic::getStackedVarArray($id);
		$c	= count($a);
		if ($c < 1) {
			trigger_error('StackedVar: no var to set', E_USER_WARNING);
			return 0;
		}
		$a[$c-1] =& $value;
	}

	function setStackedVar($id, $value) {
		Logic::setStackedVarRef($id, $value);
	}
}

/**
 * Implementation of common JSTL functions.
 *  @see http://java.sun.com/webservices/docs/1.2/tutorial/doc/JSTL4.html
 */
class LogicTagImpl {

	function getScopeVariableName($scope) {
		switch($scope) {
			case '':
			case 'page':
				return 'pageScope';
			case 'request':
				return 'requestScope';
			case 'session':
				return 'sessionScope';
			case 'application':
				return 'applicationScope';
		}
		return '';
	}

	function c_set($parameterMap) {
		$var = (isset($parameterMap['var']) ? $parameterMap['var'] : '');
		$scope = (isset($parameterMap['scope']) ? $parameterMap['scope'] : '');
		$parent = LogicTagImpl::getScopeVariableName($scope);
		if (isset($parameterMap['value'])) {
			Logic::set($parent, $var, $parameterMap['value']);
		} else {
			Logic::setProperty($parent, $var, $parameterMap['data']);
		}
	}

	function c_remove($parameterMap) {
		$var = (isset($parameterMap['var']) ? $parameterMap['var'] : '');
		$scope = (isset($parameterMap['scope']) ? $parameterMap['scope'] : '');
		$parent = LogicTagImpl::getScopeVariableName($scope);
		Logic::remove($parent, $var);
	}

	function c_out($parameterMap) {
		$value	= (isset($parameterMap['value']) ? $parameterMap['value'] : '');
		$result	= Logic::toString($value);
		if ((isset($parameterMap['escapeXml'])) && (Logic::toBool($parameterMap['escapeXml']))) {
			$result	= strtr($result, array('&' => '&amp;', '<' => '&lt;', '>' => '&gt;', '\'' => '&#039;', '"' => '&#034;'));
		}
		return $result;
	}

	function c_if($parameterMap) {
		$test = (isset($parameterMap['test']) ? $parameterMap['test'] : '');
		if (Logic::toBool($test)) {
			return $parameterMap['data'];
		} else {
			return '';
		}
	}

	function c_choose($parameterMap) {
		$triggered	= FALSE;
		Logic::pushStackedVar('c_choose_triggered', $triggered);
		$result	= mosMoslateAPI::parse($parameterMap['data']);
		Logic::popStackedVar('c_choose_triggered');
		return $result;
	}

	function c_when($parameterMap) {
		Logic::logDebug('c_when: begin');
		$test = (isset($parameterMap['test']) ? $parameterMap['test'] : '');
		if ((!Logic::peekStackedVar('c_choose_triggered')) && (Logic::evalELAsBool($test))) {
			Logic::logDebug('c_when: triggered=TRUE');
			Logic::setStackedVar('c_choose_triggered', TRUE);
			Logic::logDebug('c_when: end');
			return $parameterMap['data'];
		} else {
			Logic::logDebug('c_when: end (empty)');
			return '';
		}
	}

	function c_otherwise($parameterMap) {
		if (!Logic::peekStackedVar('c_choose_triggered')) {
			return $parameterMap['data'];
		} else {
			return '';
		}
	}

	function c_forEach($parameterMap) {
		global $Logic_debug;
		$key = (isset($parameterMap['key']) ? $parameterMap['key'] : '');	// non-standard
		$var = (isset($parameterMap['var']) ? $parameterMap['var'] : '');
		$scope = (isset($parameterMap['scope']) ? $parameterMap['scope'] : '');
		$parent = LogicTagImpl::getScopeVariableName($scope);
		if (isset($parameterMap['items'])) {
			$list = Logic::toArray($parameterMap['items']);
		} else {
			$name = (isset($parameterMap['name']) ? $parameterMap['name'] : '');
			if ($name == '') {
				$list	= NULL;
			} else {
				$list = Logic::toArray(Logic::getBean($name, FALSE));
			}
		}
		$result	= '';
		$item	= NULL;
		if (is_array($list)) {
			foreach(array_keys($list) as $k) {
				if (isset($item)) {
					unset($item);
				}
				$item	=& $list[$k];
				if ($key != '') {
					Logic::setProperty($parent, $key, $k);
					if ($var != '') {
						Logic::set($parent, $var, $item);
					}
				} else {
					if ($var != '') {
						$mapEntry	= array();
						$mapEntry['key']	= $k;
						$mapEntry['value']	=& $list[$k];
						Logic::set($parent, $var, $mapEntry);
					}
				}
				$result	.= mosMoslateAPI::parse($parameterMap['data']);
			}
		} else {
			if (!is_null($list)) {
				trigger_error('forEach: no list (type='.gettype($list).(is_object($list) ? '; class='.get_class($list) : '').')', E_USER_WARNING);
			}
		}
		return $result;
	}

	function addUrlParameters($url, $params) {
		if (count($params) > 0) {
			$prefix	= (strpos($url, '?') !== FALSE ? '&' : '?');
			foreach($params as $k => $v) {
				if ($k != '') {
					$url	.= $prefix.urlencode($k).'='.urlencode($v);
					$prefix	= '&';
				}
			}
		}
		return $url;
	}

	function getFileExt($path) {
		$i	= strpos($path, '?');
		if ($i !== FALSE) {
			$path	= substr($path, 0, $i);
		}
		$i	= strpos($path, '.');
		if ($i !== FALSE) {
			return substr($path, $i);
		}
		return '';
	}

	function getAbsolutePath($path, $forceDynamic = FALSE) {
		global $mosConfig_absolute_path, $mosConfig_live_site;

		if (($path != '') && (strpos($path, '://') === FALSE)) {
			if (substr($path, 0, 1) != '/') {
				$path	= '/'.$path;
			}
			$ext	= strtolower(LogicTagImpl::getFileExt($path));
			if (($forceDynamic) || (strpos($path, '?') !== FALSE) ||
					($ext == '.php') || ($ext == '.jsp') || ($ext == '.asp') || ($ext == '.shtml')) {
				$path	= $mosConfig_live_site.$path;
			} else {
				$path	= $mosConfig_absolute_path.$path;
			}
		}
		return $path;
	}

	function getFileContents($path, $resolve = TRUE) {
		global $mosConfig_absolute_path, $mosConfig_live_site;

		if ($path != '') {
			if (($resolve) && (strpos($path, '://') === FALSE)) {
				if (substr($path, 0, 1) != '/') {
					$path	= '/'.$path;
				}
				$ext	= strtolower(LogicTagImpl::getFileExt($path));
				if ((strpos($path, '?') !== FALSE) ||
						($ext == '.jsp') || ($ext == '.asp') || ($ext == '.shtml')) {
					$path	= $mosConfig_live_site.$path;
				} else {
					$path	= $mosConfig_absolute_path.$path;
					if (($ext == '.php') && (file_exists($path))) {
						// use internal short-cut
						ob_start();
						include($path);
						$text	= ob_get_contents();
						ob_end_clean();
						return $text;
					}
				}
			}
			$fileHandle = fopen($path, 'rb');
			if ($fileHandle === FALSE) {
				return '';
			}
			if ((strpos($path, '://') === FALSE) && (file_exists($path))) {
				$size	= filesize($path);
				if ($size > 0) {
					$text	= fread($fileHandle, $size);
				} else {
					$text	= '';
				}
			} else {
				$text	= '';
				while (!feof($fileHandle)) {
					$text	.= fread($fileHandle, 1024);
				}
			}
			fclose($fileHandle);
		} else {
			$text	= '';
		}
		return $text;
	}

	function c_import($parameterMap) {
		$url	= (isset($parameterMap['url']) ? $parameterMap['url'] : '');
		$var = (isset($parameterMap['var']) ? $parameterMap['var'] : '');
		$scope = (isset($parameterMap['scope']) ? $parameterMap['scope'] : '');
		$parent = LogicTagImpl::getScopeVariableName($scope);
		Logic::pushStackedVar('c_param', array());
		$data	= trim(mosMoslateAPI::parse($parameterMap['data']));
		$params	=& Logic::popStackedVarRef('c_param');
		$url	= LogicTagImpl::addUrlParameters($url, $params);
		$text	= LogicTagImpl::getFileContents($url);
		if ($var != '') {
			Logic::set($parent, $var, $text);
		} else {
			return $text;
		}
	}

	function c_url($parameterMap) {
		$value	= (isset($parameterMap['value']) ? $parameterMap['value'] : '');
		$var = (isset($parameterMap['var']) ? $parameterMap['var'] : '');
		$scope = (isset($parameterMap['scope']) ? $parameterMap['scope'] : '');
		$parent = LogicTagImpl::getScopeVariableName($scope);
		Logic::pushStackedVar('c_param', array());
		$data	= trim(mosMoslateAPI::parse($parameterMap['data']));
		$params	=& Logic::popStackedVarRef('c_param');
		$url	= LogicTagImpl::addUrlParameters($value, $params);
		if ($var != '') {
			Logic::set($parent, $var, $url);
		} else {
			return $url;
		}
	}

	function c_param($parameterMap) {
		$params	=& Logic::peekStackedVarRef('c_param');
		$name = (isset($parameterMap['name']) ? $parameterMap['name'] : '');
		if ($name != '') {
			if (!isset($parameterMap['value'])) {
				$value	= $parameterMap['data'];
			} else {
				$value	= $parameterMap['value'];
			}
			$params[$name]	= $value;
		}
	}

	function sql_queryOrUpdate($parameterMap, $sqlQuery = TRUE) {
		global $database;
		$scope = (isset($parameterMap['scope']) ? $parameterMap['scope'] : '');
		$parent = LogicTagImpl::getScopeVariableName($scope);
		Logic::pushStackedVar('sql_param', array());
		$data	= trim(mosMoslateAPI::parse($parameterMap['data']));
		$params	=& Logic::popStackedVarRef('sql_param');
		if (!isset($parameterMap['sql'])) {
			$query	= $data;
		} else {
			$query	= $parameterMap['sql'];
		}
		if ((!isset($parameterMap['var'])) || ($parameterMap['var'] == '')) {
			if ($sqlQuery) {
				$var	= 'rows';
			} else {
				$var	= '';
			}
		} else {
			$var	= $parameterMap['var'];
		}
		$startPos	= 0;
		for($iParam=0; $iParam < count($params); $iParam++) {
			$param	=& $params[$iParam];
			//$text, $startStr, $endStr, $escapeStr = '', $startPos
			$i	= Logic::findTermEndSkipQuotes($query, '', '?', '', $startPos);
			if ($i !== FALSE) {
				if (is_numeric($param)) {
					$v	= floatval($param);
				} else {
					$v	= mosMoslateAPI::dbQuote($database, Logic::toString($param));
				}
				$query		= substr($query, 0, $i).$v.substr($query, $i+1);
				$startPos	= $i + strlen($v);
			} else {
				trigger_error('Logic: parameter placeholder '.$iParam.' not found', E_USER_WARNING);
				break;
			}
		}
		// TODO replace ? with parameters
		if ($sqlQuery) {
			$allowedStatements	= array('SELECT');
		} else {
			$allowedStatements	= array('INSERT', 'UPDATE', 'DELETE');
		}
		$allowed	= FALSE;
		foreach($allowedStatements as $statement) {
			if ((strlen($query) >= strlen($statement)) && (strtoupper(substr($query, 0, strlen($statement))) == $statement)) {
				$allowed	= TRUE;
				break;
			}
		}
		if (!$allowed) {
			trigger_error('Logic: query not allowed', E_USER_WARNING);
			if ($var != '') {
				$result	= FALSE;
				Logic::set($parent, $var, $result);
			}
			return;
		}
		$database->setQuery($query);
		if ($sqlQuery) {
			$rows = $database->loadObjectList();
			if ($database->getErrorNum()) {
				$msg	= 'database error:'.stripslashes($database->getErrorMsg());
				trigger_error($msg, E_USER_WARNING);
				$rows	= FALSE;
			}
			Logic::set($parent, $var, $rows);
		} else {
			$affectedRows	= -1;
			if ($database->query() !== FALSE) {
				if ((isset($database->_resource)) && (is_object($database->_resource)) && (method_exists($database->_resource, 'Affected_Rows'))) {
					$affectedRows	= $database->_resource->Affected_Rows();	// not tested yet
				} else {
					$affectedRows	= mysql_affected_rows();
				}
			} else {
				trigger_error('Logic: failed to execute query - '.$database->getErrorMsg(), E_USER_WARNING);
			}
			if ($var != '') {
				Logic::set($parent, $var, $affectedRows);
			}
		}
	}

	function sql_query($parameterMap) {
		LogicTagImpl::sql_queryOrUpdate($parameterMap, TRUE);
	}

	function sql_update($parameterMap) {
		LogicTagImpl::sql_queryOrUpdate($parameterMap, FALSE);
	}

	function sql_param($parameterMap) {
		$params	=& Logic::peekStackedVarRef('sql_param');
		if (!isset($parameterMap['value'])) {
			$value	= $parameterMap['data'];
		} else {
			$value	= $parameterMap['value'];
		}
		$params[]	= $value;
	}

	function jsp_useBean($parameterMap) {
		$id = (isset($parameterMap['id']) ? $parameterMap['id'] : 'bean');
		$class = (isset($parameterMap['class']) ? $parameterMap['class'] : '');
		$scope = (isset($parameterMap['scope']) ? $parameterMap['scope'] : '');
		$parent = LogicTagImpl::getScopeVariableName($scope);
		if ($class != '') {
			$obj =& Logic::get($parent, $id);
			if (!is_null($obj)) {
				if (!is_a($obj, $class)) {
					Logic::set($parent, $id, Logic::newInstance($class));
				}
			} else {
				Logic::set($parent, $id, Logic::newInstance($class));
			}
		}
	}

	function jsp_setProperty($parameterMap) {
		$property = (isset($parameterMap['property']) ? $parameterMap['property'] : '');
		$name = (isset($parameterMap['name']) ? $parameterMap['name'] : '');
		if (isset($parameterMap['value'])) {
			Logic::set($name, $property, $parameterMap['value']);
		} else {
			Logic::setProperty($name, $property, trim($parameterMap['data']));
		}
	}

	function jsp_getProperty($parameterMap) {
		$property = (isset($parameterMap['property']) ? $parameterMap['property'] : '');
		$name = (isset($parameterMap['name']) ? $parameterMap['name'] : '');
		return Logic::getProperty($name, $property);
	}

	function jsp_include($parameterMap) {
		global $mosConfig_absolute_path;
		$page	= (isset($parameterMap['page']) ? $parameterMap['page'] : '');
		$page	= str_replace('/..', '', $page);	// only within context allowed
		$page	= str_replace('\\..', '', $page);
		if (($page != '') && (substr($page, 0, 1) != '/')) {
			$page	= '/'.$page;
		}
		$text	= LogicTagImpl::getFileContents($page);
		return $text;
	}
}

?>